from django.shortcuts import render, redirect, get_object_or_404 
from . models import (ProductList, ProductImage, 
                        CategoryList, BrandList, SizeList, ColorList, ProductWiseSize, ProductWiseCategory, 
                        ProductWiseColor, SalesOrder, SalesOrderDetails, CustomerList
                    )

from django.utils.text import slugify
from django.contrib import messages 
from hr.decorators import EmployeeLogin 

import datetime
from django.db.models import Max 

def generate_sku():
    today = datetime.date.today()
    date_part = today.strftime("%y%m%d")  # YYMMDD

    # সর্বশেষ SKU বের করা
    last_sku = ProductList.objects.aggregate(max_sku=Max('sku_code'))['max_sku']

    if last_sku and str(last_sku).startswith(date_part): 
        last_number = int(last_sku[-6:])
        new_number = last_number + 1
    else: 
        new_number = 1

    return f"{date_part}{new_number:06d}"

@EmployeeLogin
def product_add(request):
    if request.method == "POST":
        sku_code = generate_sku()

        # ------- Basic Product Info --------
        name_bangla = request.POST.get("name_bangla")
        name_english = request.POST.get("name_english") 
        category_ids = request.POST.getlist('category_id[]')
        stock_status = request.POST.get("stock_status") 
        
        is_variant = True if request.POST.get("is_variant") else False

        print("The Varient Mode is------------ : ", is_variant)

        unit_price = 0
        discount = 0
        discount_price = 0
        discount_status = 0
        quantity = 0
        if not is_variant:
            unit_price = int(request.POST.get("unit_price")) or 0
            discount = int(request.POST.get("discount")) or 0
            discount_price = int(request.POST.get("discount_price")) or 0
            quantity = request.POST.get("quantity") or 0

            discount_status = 1 if discount_price > 0 else 0
 
      
        print("unit_price: ", unit_price)
        print("discount: ", discount)
        print("discount_price: ", discount_price)
        print("quantity: ", quantity)
        print("discount_status: ", discount_status)


        full_detail = request.POST.get("full_detail")

        # ------- SEO Fields --------
        seo_meta_title = request.POST.get("seo_meta_title")
        seo_meta_description = request.POST.get("seo_meta_description")
        seo_focus_keyword = request.POST.get("seo_focus_keyword")  
 
        product_image = request.FILES.get("product_image")
        product_others_images = request.FILES.getlist("product_others_image")

        # ------- Slug Generate --------
        slug = slugify(name_english) if name_english else None

        # ------- Save Main Product --------
        product = ProductList.objects.create(
            name_bangla=name_bangla,
            name_english=name_english,
            slug=slug,
            sku_code=sku_code,
            unit_price=unit_price,
            discount=discount,
            discount_price = discount_price,
            discount_status = discount_status,
            quantity=quantity,
            stock_status=stock_status, 
            full_detail=full_detail,
            is_variant = is_variant,
            # SEO
            seo_meta_title=seo_meta_title,
            seo_meta_description=seo_meta_description,
            seo_focus_keyword=seo_focus_keyword,   
        )

        
        
        print("category_ids: ", category_ids)
        for cat_id in category_ids:
            if cat_id:
                ProductWiseCategory.objects.create(
                    product=product,
                    category=CategoryList.objects.get(id=cat_id),
                    status=True
                )
 
        # --- Color Save ---
        color_ids = request.POST.getlist('color_id[]')
        for color_id in color_ids:
            if color_id: 
                ProductWiseColor.objects.create(
                    product=product, 
                    color=ColorList.objects.get(id=color_id),
                    status=True
                )

        # --- Size Save ---
        # size_ids = request.POST.getlist('size_id[]')
        # for s_id in size_ids:
        #     if s_id:
        #         size_unit_price = int(request.POST.get(f'size_price_{s_id}'))
        #         size_discount = int(request.POST.get(f'size_discount_{s_id}'))
        #         size_quantity = int(request.POST.get(f'size_qty_{s_id}'))

        #         discount_price = 0
        #         if size_discount > 0:
        #             discount_price = size_unit_price - ((size_unit_price * size_discount) / 100)

        #         ProductWiseSize.objects.create(
        #             product=product,
        #             size=SizeList.objects.get(id=s_id), 
        #             unit_price=size_unit_price or 0,
        #             discount=size_discount or 0,
        #             quantity=size_quantity or 0,
        #             discount_price=discount_price or 0,
        #             status=True
        #         )


        size_ids = request.POST.getlist('size_id[]')

        for index, s_id in enumerate(size_ids):   # enumerate দিয়ে index ট্র্যাক করছি
            if s_id:
                size_unit_price = int(request.POST.get(f'size_price_{s_id}'))
                size_discount = int(request.POST.get(f'size_discount_{s_id}'))
                size_quantity = int(request.POST.get(f'size_qty_{s_id}'))

                discount_price = 0
                if size_discount > 0:
                    discount_price = size_unit_price - ((size_unit_price * size_discount) / 100)

                # ProductWiseSize Create
                ProductWiseSize.objects.create(
                    product=product,
                    size=SizeList.objects.get(id=s_id), 
                    unit_price=size_unit_price or 0,
                    discount=size_discount or 0,
                    quantity=size_quantity or 0,
                    discount_price=discount_price or 0,
                    status=True
                )

                #  শুধু প্রথম iteration এ ProductList update হবে
                if index == 0:
                    product.unit_price = size_unit_price or 0
                    product.discount = size_discount or 0
                    product.quantity = size_quantity or 0
                    product.discount_price = discount_price or 0
                    product.save()


 
        if product_others_images:
            for img in product_others_images:
                ProductImage.objects.create(product=product, image=img)
 
        if product_image:
            product.product_image = product_image
            product.save()

        return redirect("/inventory/product-list/")   
     
     

    context = {
        'categories': CategoryList.objects.filter(status=True),
        'brands': BrandList.objects.filter(status=True),
        'sizes': SizeList.objects.filter(status=True),
        'colors': ColorList.objects.filter(status=True),
    }
    return render(request, 'inventory/product/product_add.html', context)



# Product List
# @EmployeeLogin
# def product_list(request):  
#     products = ProductList.objects.all().order_by('-id')  

 
#     return render(request, "inventory/product/product_list.html", {'products': products})


@EmployeeLogin
def product_list(request):  
    products = ProductList.objects.all().order_by('-id')

    product_data = []
    for product in products:
        if product.is_variant:
            size = ProductWiseSize.objects.filter(product=product).first()  # Just First row

            if size:
                unit_price = size.unit_price
                discount_price = size.discount_price
                discount = size.discount
                quantity = size.quantity
            else:
                unit_price = product.unit_price
                discount_price = product.discount_price
                discount = product.discount
                quantity = product.quantity
        else:
            unit_price = product.unit_price
            discount_price = product.discount_price
            discount = product.discount
            quantity = product.quantity

        product_data.append({
            'id': product.id,
            'name_english': product.name_english,
            'sku_code': product.sku_code,
            'product_image': product.product_image,
            'unit_price': unit_price,
            'discount': discount,
            'discount_price': discount_price,
            'quantity': quantity,
            'stock_status': product.get_stock_status_display(),
            'deleted_status': product.deleted_status,
        })

    return render(request, "inventory/product/product_list.html", {
        'products': product_data
    })

# Product Varient Add
@EmployeeLogin
def product_varient_add(request, id):   
    product = get_object_or_404(ProductList, id=id)
    categories = CategoryList.objects.all().order_by("ordering")
    colors = ColorList.objects.all().order_by("ordering")
    sizes = SizeList.objects.all().order_by("ordering")
    brands = BrandList.objects.all().order_by("ordering")

    if request.method == "POST": 
        brand_id = request.POST.get('brand_id')
        if brand_id:
            product.brand_id = brand_id
            product.save()

        category_ids = request.POST.getlist('category_id[]')
        print("category_ids: ", category_ids)
        for cat_id in category_ids:
            if cat_id:
                ProductWiseCategory.objects.create(
                    product=product,
                    category=CategoryList.objects.get(id=cat_id),
                    status=True
                )
 
        # --- Color Save ---
        color_ids = request.POST.getlist('color_id[]')
        for color_id in color_ids:
            if color_id:
                unit_price = request.POST.get(f'color_price_{color_id}')
                discount_price = request.POST.get(f'color_discount_{color_id}')
                color_image = request.FILES.get(f'color_image_{color_id}')
                ProductWiseColor.objects.create(
                    product=product,
                    color=ColorList.objects.get(id=color_id),
                    unit_price=unit_price or 0,
                    discount_price=discount_price or 0,
                    # color_image=color_image,
                    status=True
                )

        # --- Size Save ---
        size_ids = request.POST.getlist('size_id[]')
        for s_id in size_ids:
            if s_id:
                ProductWiseSize.objects.create(
                    product=product,
                    size=SizeList.objects.get(id=s_id),
                    status=True
                )

        messages.success(request, "Product Variant saved successfully!")
        return redirect('product_list')   
    

    context = {
        'categories': categories,
        'colors': colors,
        'sizes': sizes,
        'brands': brands,
    }
    return render(request, "inventory/product/product_varient_add.html", context)


from django.http import JsonResponse
from django.db.models import Q 

def product_search(request):
    query = request.GET.get("q", "").strip()

    print("query: ", query)
    products = []

    if query:
        products = ProductList.objects.filter(
            Q(name_english__icontains=query) |
            Q(name_bangla__icontains=query) |
            Q(sku_code__icontains=query) 
        )[:10]

        print("products: ", products)

    data = []
    for p in products:
        data.append({
            "id": p.id,
            "name": p.name_english or p.name_bangla,
            "sku": p.sku_code,
            "price": float(p.unit_price),
            "image": p.product_image.url if p.product_image else "",
        })
    return JsonResponse(data, safe=False)

 
def sales_order_create(request): 
    
    return render(request, "inventory/product/pos_add.html")




def sales_order_list(request):
    # Start with all orders
    orders = SalesOrder.objects.select_related(
        'customer', 'delivery_method', 'district', 'upozilla'
    ).order_by('-order_date')

    # Get filter values
    customer_name = request.GET.get('customer', '').strip()
    order_id = request.GET.get('order_id', '').strip()
    order_date = request.GET.get('date', '').strip()
    status = request.GET.get('status', '').strip()

    # Apply filters
    if customer_name:
        orders = orders.filter(customer__name__icontains=customer_name)
    if order_id:
        orders = orders.filter(order_number__icontains=order_id)
    if order_date:
        orders = orders.filter(order_date__date=order_date)
    if status:
        orders = orders.filter(payment_status__icontains=status)

    context = {
        "orders": orders
    }
    return render(request, "inventory/product/sales_order_list.html", context)


def sales_order_invoice(request, pk):
    # Main order
    order = get_object_or_404(SalesOrder, pk=pk)
    
    # Related order items
    order_items = SalesOrderDetails.objects.filter(order_no=order)
    
    context = {
        "order": order,
        "order_items": order_items,
    }
    return render(request, "inventory/sales_order/invoice.html", context)



def customer_list(request):
    # Filter/Search
    query = CustomerList.objects.all()

    customer_name = request.GET.get("customer_name")
    mobile = request.GET.get("mobile")
    email = request.GET.get("email")

    if customer_name:
        query = query.filter(customer_name__icontains=customer_name)
    if mobile:
        query = query.filter(mobile__icontains=mobile)
    if email:
        query = query.filter(email__icontains=email)

    context = {
        "customers": query
    }
    return render(request, "inventory/sales_order/customer_list.html", context)



# # READ
# def category_list(request):
#     categories = models.CategoryList.objects.all().order_by("ordering")
#     return render(request, "category_list.html", {"categories": categories})

# # CREATE
# def category_create(request):
#     if request.method == "POST":
#         models.CategoryList.objects.create(
#             name=request.POST.get("name"),
#             description=request.POST.get("description"),
#             ordering=request.POST.get("ordering"),
#             status=True if request.POST.get("status") == "on" else False,
#         )
#         return redirect("category_list")
#     return render(request, "category_form.html")

# # UPDATE
# def category_update(request, pk):
#     obj = get_object_or_404(models.CategoryList, pk=pk)
#     if request.method == "POST":
#         obj.name = request.POST.get("name")
#         obj.description = request.POST.get("description")
#         obj.ordering = request.POST.get("ordering")
#         obj.status = True if request.POST.get("status") == "on" else False
#         obj.save()
#         return redirect("category_list")
#     return render(request, "category_form.html", {"obj": obj})

# # DELETE
# def category_delete(request, pk):
#     obj = get_object_or_404(models.CategoryList, pk=pk)
#     obj.delete()
#     return redirect("category_list")



# def brand_list(request):
#     brands = models.BrandList.objects.all().order_by("ordering")
#     return render(request, "brand_list.html", {"brands": brands})

# def brand_create(request):
#     if request.method == "POST":
#         models.BrandList.objects.create(
#             brand_name=request.POST.get("brand_name"),
#             details=request.POST.get("details"),
#             ordering=request.POST.get("ordering"),
#             status=True if request.POST.get("status") == "on" else False,
#         )
#         return redirect("brand_list")
#     return render(request, "brand_form.html")

# def brand_update(request, pk):
#     obj = get_object_or_404(models.BrandList, pk=pk)
#     if request.method == "POST":
#         obj.brand_name = request.POST.get("brand_name")
#         obj.details = request.POST.get("details")
#         obj.ordering = request.POST.get("ordering")
#         obj.status = True if request.POST.get("status") == "on" else False
#         obj.save()
#         return redirect("brand_list")
#     return render(request, "brand_form.html", {"obj": obj})

# def brand_delete(request, pk):
#     obj = get_object_or_404(models.BrandList, pk=pk)
#     obj.delete()
#     return redirect("brand_list")



# def color_list(request):
#     colors = models.ColorList.objects.all().order_by("ordering")
#     return render(request, "color_list.html", {"colors": colors})

# def color_create(request):
#     if request.method == "POST":
#         models.ColorList.objects.create(
#             color_name=request.POST.get("color_name"),
#             details=request.POST.get("details"),
#             ordering=request.POST.get("ordering"),
#             status=True if request.POST.get("status") == "on" else False,
#         )
#         return redirect("color_list")
#     return render(request, "color_form.html")

# def color_update(request, pk):
#     obj = get_object_or_404(models.ColorList, pk=pk)
#     if request.method == "POST":
#         obj.color_name = request.POST.get("color_name")
#         obj.details = request.POST.get("details")
#         obj.ordering = request.POST.get("ordering")
#         obj.status = True if request.POST.get("status") == "on" else False
#         obj.save()
#         return redirect("color_list")
#     return render(request, "color_form.html", {"obj": obj})

# def color_delete(request, pk):
#     obj = get_object_or_404(models.ColorList, pk=pk)
#     obj.delete()
#     return redirect("color_list")



# def size_list(request):
#     sizes = models.SizeList.objects.all().order_by("ordering")
#     return render(request, "size_list.html", {"sizes": sizes})

# def size_create(request):
#     if request.method == "POST":
#         models.SizeList.objects.create(
#             size_name=request.POST.get("size_name"),
#             short_code=request.POST.get("short_code"),
#             ordering=request.POST.get("ordering"),
#             status=True if request.POST.get("status") == "on" else False,
#         )
#         return redirect("size_list")
#     return render(request, "size_form.html")

# def size_update(request, pk):
#     obj = get_object_or_404(models.SizeList, pk=pk)
#     if request.method == "POST":
#         obj.size_name = request.POST.get("size_name")
#         obj.short_code = request.POST.get("short_code")
#         obj.ordering = request.POST.get("ordering")
#         obj.status = True if request.POST.get("status") == "on" else False
#         obj.save()
#         return redirect("size_list")
#     return render(request, "size_form.html", {"obj": obj})

# def size_delete(request, pk):
#     obj = get_object_or_404(models.SizeList, pk=pk)
#     obj.delete()
#     return redirect("size_list")
